<?php
require_once 'config.php';
require_once 'header.php';

// Redirect if not logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_project'])) {
    $project_title = trim($_POST['project_title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $domain = trim($_POST['domain'] ?? '');
    $team_members = trim($_POST['team_members'] ?? '');
    $start_date = $_POST['start_date'] ?? '';
    $end_date = $_POST['end_date'] ?? '';

    if (empty($project_title) || empty($description) || empty($domain) || empty($team_members) || empty($start_date) || empty($end_date)) {
        $error = 'All fields are required.';
    } elseif (strtotime($end_date) < strtotime($start_date)) {
        $error = 'End date cannot be before start date.';
    } else {
        $stmt = $pdo->prepare("INSERT INTO projects (user_id, project_title, description, domain, team_members, start_date, end_date) VALUES (?, ?, ?, ?, ?, ?, ?)");
        if ($stmt->execute([$user_id, $project_title, $description, $domain, $team_members, $start_date, $end_date])) {
            $success = 'Project submitted successfully.';
        } else {
            $error = 'Submission failed. Please try again.';
        }
    }
}

// Fetch user's projects
$stmt = $pdo->prepare("SELECT * FROM projects WHERE user_id = ? ORDER BY created_at DESC");
$stmt->execute([$user_id]);
$projects = $stmt->fetchAll();
?>

<h2>Welcome, <?= htmlspecialchars($_SESSION['user_name']) ?></h2>

<h3>Submit a New Project</h3>
<?php if (isset($error)): ?><p class="error"><?= htmlspecialchars($error) ?></p><?php endif; ?>
<?php if (isset($success)): ?><p class="success"><?= htmlspecialchars($success) ?></p><?php endif; ?>

<form method="post">
    <label for="project_title">Project Title:</label>
    <input type="text" name="project_title" id="project_title" required value="<?= htmlspecialchars($_POST['project_title'] ?? '') ?>">

    <label for="description">Description:</label>
    <textarea name="description" id="description" rows="5" required><?= htmlspecialchars($_POST['description'] ?? '') ?></textarea>

    <label for="domain">Domain / Category:</label>
    <input type="text" name="domain" id="domain" required value="<?= htmlspecialchars($_POST['domain'] ?? '') ?>">

    <label for="team_members">Team Members (comma-separated names):</label>
    <input type="text" name="team_members" id="team_members" required value="<?= htmlspecialchars($_POST['team_members'] ?? '') ?>">

    <label for="start_date">Start Date:</label>
    <input type="date" name="start_date" id="start_date" required value="<?= htmlspecialchars($_POST['start_date'] ?? '') ?>">

    <label for="end_date">End Date:</label>
    <input type="date" name="end_date" id="end_date" required value="<?= htmlspecialchars($_POST['end_date'] ?? '') ?>">

    <button type="submit" name="submit_project">Submit Project</button>
</form>

<h3>Your Past Projects</h3>
<?php if (count($projects) > 0): ?>
    <table>
        <thead>
            <tr>
                <th>ID</th>
                <th>Title</th>
                <th>Domain</th>
                <th>Start Date</th>
                <th>End Date</th>
                <th>Status</th>
                <th>Submitted On</th>
            </tr>
        </thead>
        <tbody>
        <?php foreach ($projects as $proj): ?>
            <tr>
                <td><?= $proj['id'] ?></td>
                <td><?= htmlspecialchars($proj['project_title']) ?></td>
                <td><?= htmlspecialchars($proj['domain']) ?></td>
                <td><?= date('d M Y', strtotime($proj['start_date'])) ?></td>
                <td><?= date('d M Y', strtotime($proj['end_date'])) ?></td>
                <td><?= ucfirst($proj['status']) ?></td>
                <td><?= date('d M Y, h:i A', strtotime($proj['created_at'])) ?></td>
            </tr>
        <?php endforeach; ?>
        </tbody>
    </table>
<?php else: ?>
    <p>You haven't submitted any projects yet.</p>
<?php endif; ?>

<?php require_once 'footer.php'; ?>